import Range
import random
import numpy as np
import re


# this range uses a sp file generated by hfss to simulate array like Babakhani
class SimulatedRange(Range.Range):
    # phase accuracy is the "maximum" error of the array phase drivess
    def __init__(self,phase_accuracy=1, verbose=False):
        self.phase_accuracy = phase_accuracy
        self.verbose = verbose
        path = "./Range/Simulation_Files/"
        file_name = "fabricated_array_5_by_1_lambda_spacing_5_by_1_with_Meta_Gaps.s138p"
        trans_ports = range(0,5)
        sw_ports = range(5,101)
        rec_ports = range(101,138)
        self.receiver_locations = range(-90,95,5)
        self.loadSPfile(path+file_name,rec_ports, trans_ports, sw_ports)
        # initialize used properties to zeros so they are easy to detect
        # but will not crash code if not setup
        self.S_ctrl = np.zeros((self.num_control_ports,self.num_control_ports),dtype=np.complex_)
        self.S_embed = np.zeros((self.num_access_ports,self.num_access_ports),dtype=np.complex_)
        self.V_drive = np.zeros((self.num_access_ports,1),dtype=np.complex_)
        self.receiver_index = 0

        self.programState(0)
        self.programPhases([0 for i in range(self.num_drive_ports)])

############### INTERFACE FUNCTIONS ###############

    def moveToTheta(self, theta):
        if self.verbose:
            print "Moving to position: theta =", theta
        self.receiver_index = self.num_drive_ports + self.receiver_locations.index(theta)

    def moveToPhi(self, phi):
        if self.verbose:
            print "Moving to position: phi =", phi

    def programState(self,state):
        if self.verbose:
            print "Programming state:", state
        self.updateController(state)
        self.embedController()

    def turnOffAllPorts(self):
        for i in range(self.num_drive_ports):
            self.V_drive[i] = 0

    def turnOnPort(self,port_idx):
        self.V_drive[port_idx] = 1

    def programPhases(self,phases):
        for i in range(self.num_drive_ports):
            #print phases[i]
            amplitude = abs(self.V_drive[i])
            self.V_drive[i] = amplitude*np.exp(1j*phases[i]*np.pi/180.0)

    def optimizePhase(self):
        if self.verbose:
            print "Optimizing Phases"
        self.turnOffAllPorts()
        optimum_phases = [0]*self.num_drive_ports
        self.turnOnPort(0)
        for i in range(1,self.num_drive_ports):
            self.turnOnPort(i)
            optimum_phases[i] = self.searchForOptPhase(i,optimum_phases)
        return optimum_phases

    def measure(self): # return absorbed power in dBm
        V_rec = np.dot(self.S_embed,self.V_drive)
        data = V_rec[self.receiver_index][0]
        power = abs(data)**2/50
        power_db = 10*np.log10(power/0.001)
        return power_db + 60

############### Child Class Helper Functions ###############

    def loadSPfile(self,file_name,rec_ports, trans_ports, sw_ports):
        self.num_control_ports = len(sw_ports)
        self.num_drive_ports = len(trans_ports)
        self.num_rec_ports = len(rec_ports)
        self.num_access_ports = self.num_drive_ports + self.num_rec_ports
        with open(file_name,'r') as file:
            line = file.readline()
            finished_preamble = False
            while not finished_preamble:
                if line:
                    if line[0] == "#":
                        finished_preamble = True
                    else:
                        line = file.readline()
            file_params = re.split(' |\r\n',line)
            self.frequency_unit = file_params[1]
            self.parameter_type = file_params[2]
            self.format = file_params[3]
            self.impedance = file_params[5]
            assert(self.parameter_type == "S")
            assert(self.format == "MA")

            num_ports = 0
            read_ports = False
            line = file.readline()
            matrix_data = []
            while not read_ports:
                line = file.readline()
                split_line = re.split('\[|\]',line)
                if len(split_line) == 1:
                    read_ports = True
                else:
                    num_ports = max(num_ports,int(split_line[1]))
            read_matrix = False
            while not read_matrix:
                split_line = re.split('\s*',line)
                if split_line[0] == '!':
                    read_matrix = True
                else:
                    split_line = split_line[1:]
                    numeric_data = myList = [float(value) for value in split_line if value != '']
                    matrix_data += numeric_data
                    line = file.readline()
            complex_data = []
            mag_idx = 0
            angle_idx = 1
            # convert to complex
            while angle_idx < len(matrix_data):
                magnitude = matrix_data[mag_idx]
                angle = matrix_data[angle_idx]
                complex_value = magnitude * np.exp(1j*angle*np.pi/180.0)
                complex_data += [complex_value]
                mag_idx = mag_idx + 2
                angle_idx = angle_idx + 2
            # reshape to 2D numpy matrix
            sp_matrix = np.reshape(complex_data, (num_ports,num_ports))
            # place in standard order
            standard_order = trans_ports + rec_ports + sw_ports;
            sp_transpose_col = sp_matrix[:,standard_order]
            Sp = sp_transpose_col[standard_order,:]
            # extract sub matrixes
            # m is the number of accessible ports
            m = self.num_access_ports
            # n is the number of embedded ports (controller ports)
            n = self.num_control_ports
            # S = [S_mm, S_mn; S_nm, S_nn]
            self.S_mm = Sp[0:m,0:m]
            self.S_mn = Sp[0:m,m:(m+n+1)]
            self.S_nm = Sp[m:(m+n+1),0:m]
            self.S_nn = Sp[m:(m+n+1),m:(m+n+1)]

    def embedController(self):
        #self.S_embed = self.S_mm + self.S_mn * self.S_ctrl*(I - self.S_nn*self.S_ctrl)^-1 * self.S_nm
        I = np.identity(self.num_control_ports)
        a = np.dot(self.S_nn,self.S_ctrl)
        b = I - a
        c = np.linalg.inv(b)
        d = np.dot(self.S_ctrl,c)
        e = np.dot(d,self.S_nm)
        f = np.dot(self.S_mn,e)
        self.S_embed = f + self.S_mm

    def updateController(self, state):
        # convert state number to reflection coefficients
        sw_states = []
        for sw_idx in range(24):
            # is switch on (1) or off (0)
            sw_binary = (state>>sw_idx) & 1
            # S = 1 for off switch, -1 for on switch
            sw_state = -2*sw_binary+1
            sw_states += [sw_state]
        # repeat switch states for each mg sheet
        sw_states = sw_states*4
        sw_states = np.array(sw_states,dtype=np.complex_)
        # controller is diagonal so there is no coupling between ports
        self.S_ctrl = np.diag(sw_states)


    def searchForOptPhase(self,tile_idx,optimized_tile_phases):
        optimized_tile_phases[tile_idx] = 0
        self.programPhases(optimized_tile_phases)
        meas_0 = self.measure()
        optimized_tile_phases[tile_idx] = 120
        self.programPhases(optimized_tile_phases)
        meas_120 = self.measure()
        optimized_tile_phases[tile_idx] = 240
        self.programPhases(optimized_tile_phases)
        meas_240 = self.measure()
        min_initial_meas = min([meas_0,meas_120,meas_240])
        # first third
        if min_initial_meas == meas_240:
            left_phase = 0
            left_meas = meas_0
            right_phase = 120
            right_meas = meas_120
        # second third
        if min_initial_meas == meas_0:
            left_phase = 120
            left_meas = meas_120
            right_phase = 240
            right_meas = meas_240
        # third third
        if min_initial_meas == meas_120:
            left_phase = 240
            left_meas = meas_240
            right_phase = 360
            right_meas = meas_0
        return self.binarySearchForOptPhase(tile_idx,optimized_tile_phases,left_phase,left_meas,right_phase,right_meas)

    def binarySearchForOptPhase(self,tile_idx,optimized_tile_phases,left_phase,left_meas,right_phase,right_meas):
        new_phase = (left_phase+right_phase)/2.0
        phase_delta = right_phase - left_phase
        # if the difference between endpoints is less than twice the accuracy tolerance
        # then the midpoint will always be at most the accuracy tolerance away from the maximum
        if phase_delta < 2*self.phase_accuracy:
            return new_phase
        optimized_tile_phases[tile_idx] = new_phase
        self.programPhases(optimized_tile_phases)
        # new meas WILL be greater that sides due to sinusoidal nature (ignoring noise)
        new_meas = self.measure()
        # replace the minimum with the new phase
        if left_meas > right_meas:
            return self.binarySearchForOptPhase(tile_idx,optimized_tile_phases,left_phase,left_meas,new_phase,new_meas)
        else:
            return self.binarySearchForOptPhase(tile_idx,optimized_tile_phases,new_phase,new_meas,right_phase,right_meas)
